<?php

namespace Convoworks\League\Plates\Extension;

use Convoworks\League\Plates\Engine;
use Convoworks\League\Plates\Template\Template;
use LogicException;
/**
 * Extension that adds the ability to create "cache busted" asset URLs.
 */
class Asset implements ExtensionInterface
{
    /**
     * Instance of the current template.
     * @var Template
     */
    public $template;
    /**
     * Path to asset directory.
     * @var string
     */
    public $path;
    /**
     * Enables the filename method.
     * @var boolean
     */
    public $filenameMethod;
    /**
     * Create new Asset instance.
     * @param string  $path
     * @param boolean $filenameMethod
     */
    public function __construct($path, $filenameMethod = \false)
    {
        $this->path = \rtrim($path, '/');
        $this->filenameMethod = $filenameMethod;
    }
    /**
     * Register extension function.
     * @param Engine $engine
     * @return null
     */
    public function register(Engine $engine)
    {
        $engine->registerFunction('asset', array($this, 'cachedAssetUrl'));
    }
    /**
     * Create "cache busted" asset URL.
     * @param  string $url
     * @return string
     */
    public function cachedAssetUrl($url)
    {
        $filePath = $this->path . '/' . \ltrim($url, '/');
        if (!\file_exists($filePath)) {
            throw new LogicException('Unable to locate the asset "' . $url . '" in the "' . $this->path . '" directory.');
        }
        $lastUpdated = \filemtime($filePath);
        $pathInfo = \pathinfo($url);
        if ($pathInfo['dirname'] === '.') {
            $directory = '';
        } elseif ($pathInfo['dirname'] === \DIRECTORY_SEPARATOR) {
            $directory = '/';
        } else {
            $directory = $pathInfo['dirname'] . '/';
        }
        if ($this->filenameMethod) {
            return $directory . $pathInfo['filename'] . '.' . $lastUpdated . '.' . $pathInfo['extension'];
        }
        return $directory . $pathInfo['filename'] . '.' . $pathInfo['extension'] . '?v=' . $lastUpdated;
    }
}
