<?php

declare (strict_types=1);
namespace Convo\Core\Factory;

use Convo\Core\ComponentNotFoundException;
use Convo\Core\ConvoServiceInstance;
use Convo\Core\Expression\EvaluationContext;
use Convo\Core\IAdminUser;
use Convo\Core\Publish\IPlatformPublisher;
use Convo\Core\Intent\IntentModel;
use Convo\Core\Intent\EntityModel;
use Convo\Core\ISecretStore;
use Convo\Core\IServiceDataProvider;
use Convo\Core\Migrate\AbstractMigration;
use Convo\Core\Util\IServerVarsResolver;
use Convo\Core\Workflow\IRunnableBlock;
use Convo\Core\Workflow\IServiceContext;
use Psr\Log\LoggerInterface;
class ConvoServiceFactory
{
    public const SERVICE_VERSION_ATTRIBUTE = 'convo_service_version';
    public const SERVICE_VERSION = 40;
    /**
     * @var PackageProviderFactory
     */
    private $_packageProviderFactory;
    /**
     * @var LoggerInterface
     */
    private $_logger;
    /**
     * @var IServiceDataProvider
     */
    private $_convoServiceDataProvider;
    /**
     * @var ISecretStore
     */
    private $_serviceSecretStore;
    /**
     * @var IServerVarsResolver
     */
    private $_serverVarsResolver;
    public function __construct(LoggerInterface $logger, \Convo\Core\Factory\PackageProviderFactory $packageProviderFactory, IServiceDataProvider $convoServiceDataProvider, ISecretStore $serviceSecretStore, IServerVarsResolver $serverVarsResolver)
    {
        $this->_logger = $logger;
        $this->_packageProviderFactory = $packageProviderFactory;
        $this->_convoServiceDataProvider = $convoServiceDataProvider;
        $this->_serviceSecretStore = $serviceSecretStore;
        $this->_serverVarsResolver = $serverVarsResolver;
    }
    /**
     * @param IAdminUser $user
     * @param string $serviceId
     * @param string $versionId
     * @return ConvoServiceInstance
     */
    public function getService(IAdminUser $user, $serviceId, $versionId, $convoServiceParamsFactory)
    {
        $this->_logger->info('Creating service [' . $serviceId . '][' . $versionId . ']');
        $data = $this->_convoServiceDataProvider->getServiceData($user, $serviceId, $versionId);
        $this->_logger->debug('Data loaded');
        $provider = $this->_packageProviderFactory->getProviderFromPackageIds($data['packages']);
        $eval = new EvaluationContext($this->_logger, $provider);
        $service = new ConvoServiceInstance($this->_logger, $eval, $convoServiceParamsFactory, $this->_serviceSecretStore, $this->_serverVarsResolver, $serviceId);
        $service->setVariables($data['variables']);
        $service->setPackageIds($data['packages']);
        // INTENTS
        if (isset($data['intents'])) {
            foreach ($data['intents'] as $intent_data) {
                $intent = new IntentModel();
                $intent->load($intent_data);
                $service->addIntent($intent);
            }
        }
        // ENTITIES
        if (isset($data['entities'])) {
            foreach ($data['entities'] as $entity_data) {
                $entity = new EntityModel();
                $entity->load($entity_data);
                $service->addEntity($entity);
            }
        }
        foreach ($data['contexts'] as $context) {
            /** @var array $context */
            /** @var IServiceContext $component */
            $component = $provider->createComponent($service, $context);
            $service->addEvalContext($component);
        }
        foreach ($data['blocks'] as $block) {
            /** @var array $block */
            /** @var IRunnableBlock $component */
            $component = $provider->createComponent($service, $block);
            $service->addBlock($component);
        }
        foreach ($data['fragments'] as $fragment) {
            $service->addFragments($provider->createComponent($service, $fragment));
        }
        return $service;
    }
    public function getVariantVersion(IAdminUser $user, $serviceId, $platformId, $variant)
    {
        if ($variant === IPlatformPublisher::RELEASE_TYPE_DEVELOP) {
            return $variant;
        }
        $meta = $this->_convoServiceDataProvider->getServiceMeta($user, $serviceId);
        if (!isset($meta['release_mapping'][$platformId])) {
            throw new ComponentNotFoundException('No release definition for service [' . $serviceId . '] platform [' . $platformId . ']');
        }
        $platform_data = $meta['release_mapping'][$platformId];
        if (!isset($platform_data[$variant])) {
            throw new ComponentNotFoundException('No release definition for service [' . $serviceId . '] platform [' . $platformId . '] variant [' . $variant . ']');
        }
        if ($platform_data[$variant]['type'] === IPlatformPublisher::MAPPING_TYPE_DEVELOP) {
            return IPlatformPublisher::MAPPING_TYPE_DEVELOP;
        }
        if ($platform_data[$variant]['type'] === IPlatformPublisher::MAPPING_TYPE_RELEASE) {
            $release = $this->_convoServiceDataProvider->getReleaseData($user, $serviceId, $platform_data[$variant]['release_id']);
            return $release['version_id'];
        }
        throw new \Exception('Not expected type [' . $platform_data[$variant]['type'] . ']');
    }
    private function _getServiceVersion($servivceData)
    {
        if (!isset($servivceData[\Convo\Core\Factory\ConvoServiceFactory::SERVICE_VERSION_ATTRIBUTE])) {
            return 0;
        }
        if (\is_int($servivceData[\Convo\Core\Factory\ConvoServiceFactory::SERVICE_VERSION_ATTRIBUTE])) {
            return $servivceData[\Convo\Core\Factory\ConvoServiceFactory::SERVICE_VERSION_ATTRIBUTE];
        }
        throw new \Exception('Invalid service version [' . $servivceData[\Convo\Core\Factory\ConvoServiceFactory::SERVICE_VERSION_ATTRIBUTE] . ']');
    }
    /**
     * Will traverze through service data and set _component_id if it is missing on particular component.
     * @param array $serviceData
     */
    public function fixComponentIds(&$serviceData)
    {
        \array_walk($serviceData, function (&$item) {
            if (\is_array($item) && isset($item['class']) && (!isset($item['properties']['_component_id']) || empty($item['properties']['_component_id']))) {
                $new_id = self::generateId();
                $this->_logger->debug('Setting missing _component_id [' . $new_id . ']');
                $item['properties']['_component_id'] = $new_id;
            }
            if (\is_array($item)) {
                $this->fixComponentIds($item);
            }
        });
    }
    /**
     * Generates component id
     * @return string
     */
    public static function generateId()
    {
        return \strtolower(\bin2hex(\random_bytes(4))) . '-' . \strtolower(\bin2hex(\random_bytes(2))) . '-' . \strtolower(\bin2hex(\random_bytes(2))) . '-' . \strtolower(\bin2hex(\random_bytes(2))) . '-' . \strtolower(\bin2hex(\random_bytes(6)));
    }
    public function migrateService(IAdminUser $user, $serviceId, $provider)
    {
        $data = $provider->getServiceData($user, $serviceId, IPlatformPublisher::MAPPING_TYPE_DEVELOP);
        $config = $provider->getServicePlatformConfig($user, $serviceId, IPlatformPublisher::MAPPING_TYPE_DEVELOP);
        $meta = $provider->getServiceMeta($user, $serviceId, IPlatformPublisher::MAPPING_TYPE_DEVELOP);
        $version = $this->_getServiceVersion($data);
        if ($version < self::SERVICE_VERSION) {
            $this->_logger->debug('Migrating service from [' . $version . '] to [' . self::SERVICE_VERSION . ']');
            $migrations = $this->_getMigrationsFrom($version);
            foreach ($migrations as $migration) {
                $migration->setLogger($this->_logger);
                $this->_logger->debug('Migrating with [' . $migration . ']');
                $data = $migration->migrate($data);
                $config = $migration->migrateConfig($config);
                $meta = $migration->migrateMeta($meta);
            }
            $this->_logger->debug('Saving migrated service');
            $provider->saveServiceData($user, $serviceId, $data);
            $provider->updateServicePlatformConfig($user, $serviceId, $config);
            $provider->saveServiceMeta($user, $serviceId, $meta);
        }
    }
    /**
     * @param int $version
     * @return AbstractMigration[]
     */
    private function _getMigrationsFrom($version)
    {
        $migrations = [];
        $all = $this->_getAllMigrations();
        foreach ($all as $migration) {
            if ($migration->getVersion() > $version && $migration->getVersion() <= self::SERVICE_VERSION) {
                $migrations[] = $migration;
            }
        }
        return $migrations;
    }
    /**
     * @return AbstractMigration[]
     */
    private function _getAllMigrations()
    {
        $migrations = [];
        $migrations[] = new \Convo\Core\Migrate\MigrateTo29();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo30();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo31();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo32();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo33();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo34();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo35();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo36();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo37();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo38();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo39();
        $migrations[] = new \Convo\Core\Migrate\MigrateTo40();
        return $migrations;
    }
    // UTIL
    public function __toString()
    {
        return \get_class($this) . '[]';
    }
}
